//==========================================================================
// EliMZ_TextWindow.js
//==========================================================================

/*:
@target MZ
@base EliMZ_Book
@orderAfter EliMZ_AnimatedFaces
@orderAfter EliMVZ_AnimatedFaces

@plugindesc ♦2.0.3♦ - Dynamically adds text windows into the screen!
@author Hakuen Studio
@url https://hakuenstudio.itch.io/hakuen-studio-text-window-for-rpg-maker-mv-mz/rate?source=game

@help
★★★★★ Rate the plugin! Please, is very important to me ^^

● Terms of Use
https://www.hakuenstudio.com/terms-of-use-5-0-0

============================================================================
Requirements
============================================================================

Need Eli Book.
Order After Eli Animated Faces.

============================================================================
Features
============================================================================

● Adds multiple text windows on the screen.
● Text windows can be moved, opened, and closed with easing animations!
● Write the text instantly, or character by character, like the default 
message window!
● Play a sound to open/close the window!
● Use faces or character sprite images to be the face of the message!
● Create template settings for quick use inside the game!
● Manipulate Static Windows on screen, changing its message and positions 
on the fly!

============================================================================
How to use
============================================================================

https://docs.google.com/document/d/1SM7UsQ-Wcnny1Akw5znBfwZXylL2Wp2kxhj-YopqrfA/edit?usp=sharing

============================================================================

@param templates
@text Window Templates
@type struct<templateSt>[]
@desc A list of templates settings to quickly show a text window in-game.
@default 

@command cmd_setupAndShow
@text Setup and Show
@desc Show a text window using a template.

    @arg template
    @text Temp Template
    @type struct<templateSt>
    @desc Create a new temporary template, just like the plugin parameters.
    @default

    @arg id
    @text Template ID
    @type text
    @desc The template id. Is not case sensitive.
    @default

    @arg winId
    @text Window ID
    @type text
    @desc An unique ID for this window. So you can manipulate it with other plugin commands. Not case-sensitive.
    @default

    @arg messages
    @text Messages
    @type struct<messageSt>[]
    @desc Setup the messages and face image that will be shown on the text window.
    @default []

@command cmd_staticWindow
@text Static Window
@desc Select a static window created with the "Setup And Show" command, and change it's properties.

    @arg winId
    @text Window ID
    @type text
    @desc An unique ID for this window. So you can manipulate it with other plugin commands. Not case-sensitive.
    @default

    @arg action
    @text Action
    @type select
    @option Open
    @option Close
    @option Update
    @desc "Close" and "Open" will play the show/hide window animation.
    @default Update

    @arg opacity
    @text Opacity
    @type text
    @desc Change the opacity of the window. Leave it blank to not change.
    @default

    @arg positionOperation
    @text Position Operation
    @type combo
    @option Add
    @option Set
    @desc Change the X position.
    @default Set

    @arg x
    @text Position X
    @type text
    @desc Change the X position. Leave it blank to not change.
    @default 
    @parent positionOperation

    @arg y
    @text Position Y
    @type text
    @desc Change the Y position. Leave it blank to not change.
    @default
    @parent positionOperation

    @arg messages
    @text Add new Messages
    @type struct<messageSt>[]
    @desc Add new messages to that window.
    @default []

    @arg index
    @text Message Index
    @type combo
    @option Next
    @option Previous
    @option Last
    @option None
    @desc Choose if you want the window to write the Next, or Previous, or Last message. Or set an index number.
    @default None
    @parent messages

@command cmd_refreshStatic
@text Refresh Static Window
@desc Force a static window refresh. Useful if you don't want to wait for the update interval.

    @arg winId
    @text Window ID
    @type text
    @desc An unique ID for this window. So you can manipulate it with other plugin commands. Not case-sensitive.
    @default

*/

/* -------------------------------- TEMPLATES ------------------------------- */
{
/*~struct~templateSt:

@param id
@text Template ID
@type text
@desc The template id. It is not case sensitive.
@default MyTemplate

@param contents
@text Contents
@type struct<contentsSt>
@desc The settings for the text and face images of this window content.
@default 

@param win
@text Window
@type struct<winSt>
@desc Settings for the text window.
@default 

@param isStatic
@text Static Window
@type boolean
@desc If true, the window will remain on screen, not playing the hide animation.
@default false

@param updateInterval
@text Update Interval
@type combo
@option Manual
@desc Only works for static windows. Sets the time in frames, where the window will update the text.
@default 60
@parent isStatic

*/

}

/* --------------------------------- CONTENT -------------------------------- */
{
/*~struct~contentsSt:

@param textOffsetX
@text Text OffsetX
@type text
@desc An extra indentation for the start position X of the text in the window.
@default 0

@param writeType
@text Write Type
@type combo
@option Message
@option Instantly
@desc "Message" will write the text like the message window.
@default Instantly

@param imageType
@text Image Type
@type combo
@option Face
@option Character
@option None
@desc The type of image that will be used as the face of the message.
@default Face

@param imageAlignY
@text Image Alignment Y
@type combo
@option top
@option center
@option bottom
@desc How the image will be positioned on the window.
@default center
@parent imageType

@param imageLayer
@text Image Layer
@type combo
@option outside
@option inside
@desc If the image will be inside the message window or outside of it, above the frame.
@default inside
@parent imageType

*/
}

/* ----------------------------- WINDOW SETTINGS ---------------------------- */
{
/*~struct~winSt:

@param skin
@text Skin File
@type file
@dir img/system
@desc The window skin that this message will use. Leave empty for default.
@default

@param backgroundType
@text Background Type
@type combo
@option Window
@option Dim
@option Transparent
@option Strong
@option Light Gradient Vertical
@option Faded Horizontal
@desc The background type.
@default Window

@param tone
@text Tone
@type text
@desc The window tone in RGB format. Separate each one with a comma. 
-255 to 255.
@default 0, 0, 0

@param layer
@text Layer
@type combo
@option Below Pictures
@option Above Pictures and below Windows
@option Above Windows
@desc The layer position of the window.
@default Above Windows

@param width
@text Width
@type text
@desc The window width. Leave 0 for automatic.
@default 0

@param height
@text Height
@type text
@desc The window height. Leave 0 for automatic.
@default 0

@param position
@text Position
@type struct<positionSt>
@desc The initial and target/final position of the window.
@default

@param openness
@text Open/Close Behavior
@type struct<opennessSt>
@desc The open/close behavior of the window.
@default {"widthAlign":"None","heightAlign":"None","easing":"inherit","duration":"10"}

@param sound
@text Open/Close Sound
@type struct<soundSt>
@desc The SE sound to play when closing and opening the window.
@default {"open":"{\"name\":\"\",\"volume\":\"75\",\"pitch\":\"0\",\"pan\":\"0\"}","close":"{\"name\":\"\",\"volume\":\"75\",\"pitch\":\"0\",\"pan\":\"0\"}"}

@param opacity
@text Opacity
@type text
@desc The opacity animation values: Start, Target, Duration.
@default 255, 255, 1

*/

}

/* -------------------------------- POSITION -------------------------------- */
{
/*~struct~positionSt:

@param duration
@text Move Duration
@type text
@desc The duration for the window to move from Initial to Target Position.
@default 1

@param easing
@text Easing
@type combo
@option linear @option --- In --- @option easeInQuad @option easeInCubic @option easeInQuart @option easeInQuint @option easeInSine @option easeInExpo @option easeInCirc @option easeInBack @option easeInBounce @option easeInElastic @option --- Out --- @option easeOutQuad @option easeOutCubic @option easeOutQuart @option easeOutQuint @option easeOutSine @option easeOutExpo @option easeOutCirc @option easeOutBack @option easeOutBounce @option easeOutElastic @option --- In / Out --- @option easeInOutQuad @option easeInOutCubic @option easeInOutQuart @option easeInOutQuint @option easeInOutSine @option easeInOutExpo @option easeInOutCirc @option easeInOutBack @option easeInOutBounce @option easeInOutElastic @option --- Out / In --- @option easeOutInQuad @option easeOutInCubic @option easeOutInQuart @option easeOutInQuint @option easeOutInSine @option easeOutInCirc @option easeOutInExpo @option easeOutInBack @option easeOutInBounce @option easeOutInElastic 
@desc Choose the easing type. Can use \v[id].
@default linear

@param delay
@text Stay Duration
@type text
@desc How much time, in frames, the window will be visible on screen after move to the target position.
@default 120

@param separator1
@text Initial Settings

@param initialAlignX
@text Align X
@type select
@option left
@option center
@option right
@option left_offScreen
@option right_offScreen
@desc Select none to only use offset value.
@default left
@parent separator1

@param initialOffsetX
@text Offset X
@type text
@desc The Offset X position.
@default 0
@parent separator1

@param initialAlignY
@text Align Y
@type select
@option top
@option center
@option bottom
@option top_offScreen
@option bottom_offScreen
@desc Select none to only use offset value.
@default top
@parent separator1

@param initialOffsetY
@text Offset Y
@type text
@desc The offset Y position.
@default 0
@parent separator1

@param separator2
@text Target Settings

@param targetAlignX
@text Align X
@type select
@option left
@option center
@option right
@desc Select none to only use offset value.
@default left
@parent separator2

@param targetOffsetX
@text Offset X
@type text
@desc The Offset X position.
@default 0
@parent separator2

@param targetAlignY
@text Align Y
@type select
@option top
@option center
@option bottom
@desc Select none to only use offset value.
@default top
@parent separator2

@param targetOffsetY
@text Offset Y
@type text
@desc The offset Y position.
@default 0
@parent separator2

*/
}

/* -------------------------------- OPENNESS -------------------------------- */
{
/*~struct~opennessSt:

@param widthAlign
@text Width Direction
@type select
@option None
@option Left to Right
@option Centered
@option Right to Left
@desc The direction that the window will open/close, regardless the width.
@default None

@param heightAlign
@text Height Direction
@type select
@option None
@option Top to Bottom
@option Centered
@option Bottom to Top
@desc The direction that the window will open/close, regardless the height.
@default Centered

@param easing
@text Easing
@type combo
@option inherit @option linear @option --- In --- @option easeInQuad @option easeInCubic @option easeInQuart @option easeInQuint @option easeInSine @option easeInExpo @option easeInCirc @option easeInBack @option easeInBounce @option --- Out --- @option easeOutQuad @option easeOutCubic @option easeOutQuart @option easeOutQuint @option easeOutSine @option easeOutExpo @option easeOutCirc @option easeOutBack @option easeOutBounce @option --- In / Out --- @option easeInOutQuad @option easeInOutCubic @option easeInOutQuart @option easeInOutQuint @option easeInOutSine @option easeInOutExpo @option easeInOutCirc @option easeInOutBack @option easeInOutBounce @option --- Out / In --- @option easeOutInQuad @option easeOutInCubic @option easeOutInQuart @option easeOutInQuint @option easeOutInSine @option easeOutInCirc @option easeOutInExpo @option easeOutInBack @option easeOutInBounce
@desc Choose the easing type. Can use \v[id]. "inherit" will get the same easing that was set on the position settings.
@default inherit

@param duration
@text Duration
@type text
@desc How fast the window will open/close. In frames.
@default 10

*/

}

/* ---------------------------- OPEN CLOSE SOUND ---------------------------- */
{
/*~struct~soundSt:

@param open
@text Open Sound (SE)
@type struct<soundObjSt>
@desc The SE to play when opening/showing the window.
@default {"name":"","volume":"75","pitch":"100","pan":"0"}

@param close
@text Close Sound (SE)
@type struct<soundObjSt>
@desc The SE to play when closing/hiding the window.
@default {"name":"","volume":"75","pitch":"100","pan":"0"}

*/

}

/* ----------------------------- SOUND SETTINGS ----------------------------- */
{
/*~struct~soundObjSt:

@param name
@text Filename
@type file
@dir audio/se
@desc The filename of the Se.
@default 

@param volume
@text Volume
@type number
@desc Volume. From 0 to 100.
@min 0
@max 100
@default 75

@param pitch
@text Pitch
@type number
@desc From -50 to 150.
@min -50
@max 150
@default 100

@param pan
@text Pan
@type number
@desc rom -100 to 100.
@min -100
@max 100
@default 0

*/
}

/* --------------------------------- MESSAGE -------------------------------- */
{
/*~struct~messageSt:

@param text
@text Text
@type multiline_string
@desc The text to show.
@default

@param separator1
@text Face Settings

@param faceName
@text File
@type file
@dir img/faces
@desc The face file to use.
@default
@parent separator1

@param faceIndex
@text Index
@type text
@desc The face index of the face file.
@default 0
@parent separator1

@param separator2
@text Character Settings

@param spriteId
@text Id
@type text
@desc The character id to be used on the window. See help file.
@default 0
@parent separator2

@param charName
@text File
@type file
@dir img/characters
@desc The character file to be used on the window's character.
@default
@parent separator2

@param charIndex
@text Index
@type text
@desc The character index of the window's character. Leave -1 to not change.
@default -1
@parent separator2

*/
}

"use strict"

var Eli = Eli || {}
var Imported = Imported || {}
Imported.Eli_TextWindow = true

/* ========================================================================== */
/*                                   PLUGIN                                   */
/* ========================================================================== */
{

class Container_TextWindow extends PIXI.Container{

    constructor(){
        super()
        this.width = Graphics.width
        this.height = Graphics.height
    }

    update(){
        for(const child of this.children){

            if(child.update){
                child.update()
            }
        }
    }

}

class Sprite_TextWindowFace extends Sprite {

    initialize(parameters, messageWindow){
        super.initialize()
        this.initializePlus(parameters, messageWindow)
        this.setFrame(0, 0, Eli.Utils.getFaceSize().width, Eli.Utils.getFaceSize().height)
        this.adjustPosition()
    }

    initializePlus(parameters, messageWindow){
        this.messageWindow = messageWindow
        this.parameters = parameters
        this.faceName = ""
        this.faceIndex = 0
    }

    adjustPosition(){
        const {x, y} = this.calculatePositionWithAlignment()
        this.move(x, y)
    }

    calculatePositionWithAlignment(){
        const {alignY, layer} = this.parameters.contents.image
        const msgWin = this.messageWindow
        const margin = msgWin._margin

        const heightByLayer = {
            outside: msgWin.height,
            inside: msgWin.contentsHeight(),
        }[layer]

        const xByLayer = {
            outside: msgWin.padding,
            inside: 0,
        }[layer]

        const yByLayer = {
            outside: msgWin.padding,
            inside: 0,
        }[layer]

        const yByAlign = {
            top: yByLayer,
            center: heightByLayer/2 - Eli.Utils.getFaceSize().height/2,
            bottom: heightByLayer - Eli.Utils.getFaceSize().height - yByLayer,
        }[alignY]

        return {x: xByLayer, y: yByAlign}
    }

    refreshFaceBitmap(name, index){
        this.faceName = name
        this.faceIndex = index
        this.bitmap = ImageManager.loadFace(name)
        this.bitmap.addLoadListener(() => {
            this.refreshFaceFrame(index)
        })
    }

    refreshFaceFrame(index){
        const faceWidth = Eli.Utils.getFaceSize().width
        const faceHeight = Eli.Utils.getFaceSize().height
        const rows = this.bitmap.height / faceWidth
        const cols = this.bitmap.width / faceHeight
        const x = index % cols * faceWidth
        const y = (Math.floor(index / cols) % rows) * faceHeight
    
        this.setFrame(x, y, faceWidth, faceHeight)
    }

    update(){
        super.update()
        this.visible = this.isVisible()
    }

    isVisible(){
        return this.messageWindow.imageTypeIsFace() && !!this.faceName
    }

    updateAnimation(){}
}

// Need Eli Animated Faces
class Sprite_TextWindowAnimatedFace extends Sprite{

    initialize(parameters, messageWindow){
        super.initialize()
        this.initializePlus(parameters, messageWindow)
        this.setFrame(0, 0, Eli.Utils.getFaceSize().width, Eli.Utils.getFaceSize().height)
        this.adjustPosition()
    }

    initializePlus(parameters, messageWindow){
        this.messageWindow = messageWindow
        this.parameters = parameters
        this.faceName = this.messageWindow.getCurrentMessage().faceName
        this.faceIndex = this.messageWindow.getCurrentMessage().faceIndex
        this.settings = Eli.AnimatedFaces.createEmptyFaceSetting()
        this.frameCount = 0
    }

    adjustPosition(){
        const {x, y} = this.calculatePositionWithAlignment()
        this.move(x, y)
    }

    calculatePositionWithAlignment(){
        const {alignY, layer} = this.parameters.contents.image
        const msgWin = this.messageWindow
        const margin = msgWin._margin

        const heightByLayer = {
            outside: msgWin.height,
            inside: msgWin.contentsHeight(),
        }[layer]

        const xByLayer = {
            outside: msgWin.padding,
            inside: 0,
        }[layer]

        const yByLayer = {
            outside: msgWin.padding,
            inside: 0,
        }[layer]

        const yByAlign = {
            top: yByLayer,
            center: heightByLayer/2 - Eli.Utils.getFaceSize().height/2,
            bottom: heightByLayer - Eli.Utils.getFaceSize().height - yByLayer,
        }[alignY]

        return {x: xByLayer, y: yByAlign}
    }

    refreshFaceBitmap(name, index){
        this.faceName = name
        this.faceIndex = index
        this.frameCount = 0
        this.bitmap = ImageManager.loadFace(this.faceName)
        this.bitmap.addLoadListener(() => {
            this.refreshFaceFrame()
        })
        this.settings = this.setFaceAnimatedSettings(this.faceName, this.faceIndex)
    }

    refreshFaceFrame(){
        const {width: faceWidth, height: faceHeight} = Eli.Utils.getFaceSize()
        const rows = this.bitmap.height / faceWidth
        const cols = this.bitmap.width / faceHeight
        const index = this.faceIndex
        const x = index % cols * faceWidth
        const y = (Math.floor(index / cols) % rows) * faceHeight
    
        this.setFrame(x, y, faceWidth, faceHeight)
    }

    setFaceAnimatedSettings(faceName, faceIndex){
        const getSettings = item => item.image === faceName && item.startIndex === faceIndex
        return  Eli.AnimatedFaces.param().faceSettings.find(getSettings) || 
                Eli.AnimatedFaces.createEmptyFaceSetting(faceName, faceIndex)
    }

    updateAnimation(maxIndex){
        if(this.canUpdateAnimation()){
            this.frameCount++
    
            if(this.canChangeFaceIndex()){
                this.changeFaceIndex(maxIndex)
                this.refreshFaceFrame()
                this.frameCount = 0
            }
        }
    }

    canUpdateAnimation(){
        return this.messageWindow.getCurrentMessage().faceName
    }

    canChangeFaceIndex(){
        return this.frameCount >= this.settings.frameSpeed && this.bitmap
    }

    changeFaceIndex(limitIndex){
        if(this.faceIndex >= limitIndex){
            this.faceIndex = this.settings.startIndex
        }else{
            this.faceIndex += 1
        }
    }

    update(){
        super.update()
        this.visible = this.isVisible()
    }

    isVisible(){
        return this.messageWindow.imageTypeIsFace() && !!this.faceName
    }
}

class Sprite_TextWindowCharacter extends Sprite_Character{

    initialize(character, parameters, messageWindow){
        this.initializePlus(parameters, messageWindow)
        super.initialize(character)
    }

    initializePlus(parameters, messageWindow){
        this.parameters = parameters
        this.messageWindow = messageWindow
    }

    initMembers(){
        super.initMembers()
        this.initMembersPlus()
    }

    initMembersPlus(){
        this.anchor.x = 0
        this.anchor.y = 0
    }

    update(){
        Sprite.prototype.update.call(this)

        if(this.canUpdate()){
            this.updateBitmap()
            this.updateFrame()
            this.updatePosition()
            this.updateOther()
        }

        this.updateVisibility()
    }

    updateVisibility(){
        Sprite.prototype.updateVisibility.call(this)
        this.visible = this.isVisible()
    }

    isVisible(){
        return this.messageWindow.imageTypeIsCharacter() && this._character && !this.isEmptyCharacter()
    }

    canUpdate(){
        return this._character && this.getMapSprite().bitmap && this.getMapSprite().bitmap.isReady()
    }

    updateBitmap(){
        this.bitmap = this.getMapSprite().bitmap
        this._isBigCharacter = this.getMapSprite()._isBigCharacter
    }

    updateFrame(){
        const frame = this.getMapSprite()._frame
        this.setFrame(frame.x, frame.y, frame.width, frame.height)
    }

    updatePosition(){
        this.adjustPosition()
    }

    adjustPosition(){
        const {x, y} = this.calculatePositionWithAlignment()
        this.move(x, y)
    }

    calculatePositionWithAlignment(){
        const {alignY, layer} = this.parameters.contents.image
        const msgWin = this.messageWindow
        const margin = msgWin._margin
        const frame = this._frame
        
        const heightByLayer = msgWin.contentsHeight()
        // {
        //     outside: msgWin.height,
        //     inside: msgWin.contentsHeight(),
        // }[layer]

        const xByLayer = {
            outside: msgWin.padding,
            inside: 0,
        }[layer]

        const yByLayer = {
            outside: msgWin.padding,
            inside: 0,
        }[layer]

        const yByAlign = {
            top: yByLayer,
            center: heightByLayer/2 - frame.height/2 - this._character.shiftY(),
            bottom: heightByLayer - frame.height - yByLayer,
        }[alignY]

        const jumpheight = this._character.jumpHeight()

        return {x: xByLayer, y: yByAlign - jumpheight }
    }

    getMapSprite(){
        return this._character.getMapSprite()
    }

    setMapSprite(character){}

}

class Window_TextWindow extends Window_Message {

    initialize(rect, parameters, winId){
        this.initParameters(parameters)
        super.initialize(rect, parameters)
        this.initializePlus(winId)
        this.createFaceSpriteForTextWindow()
        this.createCharSprite()

        if(this.isStaticWindow() && winId){
            Plugin.staticWindows[winId] = this
        }
    }

    get openness() {
        return this._openness
    }

    set openness(value) {
        this._openness = value.clamp(0, 255)
    }

    // Overwrite
    updateOpen() {
        if (this._opening) {
            //this.openness = 255
            if (this.isOpen()) {
                this._opening = false
            }
        }
    }

    // Overwrite
    updateClose() {
        if (this._closing) {
            //this.openness -= 32;
            if (this.isClosed()) {
                this._closing = false
            }
        }
    }

    initParameters(parameters){
        this.parameters = parameters
        this.messages = this.parameters.messages.map(item => item)
        this.parameters.messages = []
    }

    initializePlus(winId){
        this.initMembersPlus(winId)
        this.setBackgroundType(this.parameters.win.backgroundType)
    }

    initMembersPlus(winId){
        this.winId = winId
        this._isWindow = false
        this.canPlaySe = true
        this.animationGroup = new Eli.AnimeGroup([], {paused: false})
        this.character_textWindow = null
        this.cursorVisible = false
        this.isStaticReady = false
        this.isReadyToUpdate = false
        this.messageIndex = 0
        this.openness = 0
        this.stayDuration = 0
    }

    advanceMessageIndex(){
        this.messageIndex += 1
    }

    createFaceSpriteForTextWindow(){
        this.faceSprite_textWindow = this.findFaceSpriteClass()
        this.addImageSprite(this.faceSprite_textWindow)
    }

    findFaceSpriteClass(){
        if(Imported.Eli_AnimatedFaces){
            return new Sprite_TextWindowAnimatedFace(this.parameters, this)
        }else{
            return new Sprite_TextWindowFace(this.parameters, this)
        }
    }

    addImageSprite(imageSprite){
        if(this.imageLayerIsOutside()){
            this.addChild(imageSprite)
        }else{
            this.addInnerChild(imageSprite)
        }
    }

    imageLayerIsOutside(){
        return this.parameters.contents.image.layer === "outside"
    }

    imageLayerIsInside(){
        return !this.imageLayerIsOutside()
    }

    createCharSprite(){
        this.refreshCharacter_textWindow()
        this.charSprite_textWindow = new Sprite_TextWindowCharacter(this.character_textWindow, this.parameters, this)
        this.addImageSprite(this.charSprite_textWindow)
    }

    refreshCharacter_textWindow(){
        this.character_textWindow = this.findMapCharacterBySpriteId()
    }

    findMapCharacterBySpriteId(){
        const spriteId = this.parsedCharacterId(this.getCurrentMessage().spriteId)
        return Eli.Utils.getMapCharacter(spriteId)
    }

    parsedCharacterId(id){
        return Eli.Utils.convertEscapeVariablesOnly(String(id))
    }

    getCurrentMessage(){
        return this.messages[this.messageIndex]
    }

    loadWindowskin(){
        const filename = this.parameters.win.skin

        if(filename){
            this.windowskin = ImageManager.loadSystem(filename)
        }else{
            super.loadWindowskin()
        }
    }

/* --------------------------------- UPDATE --------------------------------- */

    update(){
        Window_Base.prototype.update.call(this)
        this.animationGroup.update()
        this.updateWrittingMessage()
        if(!this.canUpdateMessage()){
            this.updateOthers()
        }
    }

    updateTone(){
        const tone = this.parameters.win.tone
        this.setTone(tone[0], tone[1], tone[2])
    }

    updateWrittingMessage(){
        while(this.canUpdateMessage()){

            if(this.updateWait()) return

            if(this.updateMessage()) return
        }
    }

    updateOthers(){
        if(this.canUpdateIdleTalkAnimation()){
            this.faceSprite_textWindow.updateAnimation(this.faceSprite_textWindow.settings.middleIndex)
        }

        if(this.canUpdateStatic()){
            this.updateStaticWindow()

        }else if(!this.animationGroup.isRunning()){
            this.updateIdle() 
        }
    }

    canUpdateIdleTalkAnimation(){
        return Imported.Eli_AnimatedFaces && this.imageTypeIsFace()
    }

    canUpdateStatic(){
        return this.isStaticWindow() && this.isStaticReady
    }

    isStaticWindow(){
        return this.parameters.isStatic
    }

    updateStaticWindow(){
        if(this.stayDuration < this.getTextUpdateInterval()){
            this.stayDuration++
            
        }else{
            this.refreshStaticText()
        }
    }

    refreshStaticText(){
        this.stayDuration = 0
        this.contents.clear()
        this.isReadyToUpdate = true
        this.drawMessageInstantly(this.getCurrentMessage().text)
    }

    getTextUpdateInterval(){
        return Math.max(1, this.parameters.updateInterval)
    }

    redrawMessage(){
        this.contents.clear()
        this.isReadyToUpdate = true

        if(this.isMessageType()){
            this.startMessage(this.getCurrentMessage().text)
        }else{
            this.drawMessageInstantly(this.getCurrentMessage().text)
        }
    }

    updateIdle(){
        if(this.stayDuration < this.parameters.win.position.delay){
            this.stayDuration++

        }else if(this.stayDuration === this.parameters.win.position.delay){
            this.animationGroup.play("reverse")
            this.stayDuration++
        }
    }

    changeMessage(index){
        this.changeMessageIndex(index)
        this.refreshContents()
        this.refreshFaceSprite()
        this.redrawMessage()
    }

    changeMessageIndex(index){
        this.messageIndex = index.clamp(0, this.messages.length-1)
    }

    canUpdateMessage(){
        return this._textState && this.animationGroup.isFinished() && this.isMessageType()
    }

/* ------------------------------- END UPDATE ------------------------------- */

    startWindowActivity(){
        this.hide()
        this.stayDuration = 0
        this.isStaticReady = false 
        this.refreshContents()
        this.createAnime()
        this.playAnime()
    }

    refreshContents(){
        this.contents.clear()
        
        if(this.imageTypeIsCharacter()){
            this.refreshCharacter()
        }

        this.resizeWindowForText()
        this.refreshSideFacePosition()
        this.refreshFaceSprite()
    }

    refreshCharacter(){
        const {charName, charIndex} = this.getCurrentMessage()
        const character = this.findMapCharacterBySpriteId()

        if(this.character_textWindow !== character){
            this.character_textWindow = character
        }

        const name = Eli.Utils.convertEscapeVariablesOnly(charName) || this.character_textWindow.characterName()
        let index = Number(Eli.Utils.convertEscapeVariablesOnly(charIndex))
        index = index > -1 ? index : this.character_textWindow.characterIndex()

        this.character_textWindow.setImage(name, index)
        this.charSprite_textWindow.setCharacter(this.character_textWindow)
    }

    resizeWindowForText(){
        const rect = this.createWindowRect(this.parameters, this.getCurrentMessage())

        if(this.messageIndex > 0){
            const x = this.x + rect.width > Graphics.width ? Graphics.width - rect.width : this.x
            const y = this.y + rect.height > Graphics.height ? Graphics.height - rect.height : this.y

            this.move(x, y, rect.width, rect.height)
        }else{
            this.move(rect.x, rect.y, rect.width, rect.height)
        }
        
        this.setBackgroundType(this.parameters.win.backgroundType)
        this.createContents()
        this.cursorVisible = false
    }

    createWindowRect(parameters){
        const {width, height} = this.createWindowSize(parameters)
        const rect = new Rectangle(0, 0, Math.ceil(width), Math.ceil(height))

        return rect
    }

    createWindowSize(parameters){
        const msg = this.getCurrentMessage()
        const textSettings = this.getTextSize(msg.text)

        if(this.imageTypeIsFace()){
            var imageWidth = msg.faceName ? Eli.Utils.getFaceSize().width + 20 : 0
        }else{ 
            var imageWidth = this.character_textWindow ? this.character_textWindow.getMapSprite()._frame.width + 10 : 0
        }

        if(parameters.win.width){
            var width = parameters.win.width
        }else{
            var width = textSettings.width + this.padding * 2 + this.getItemPadding() + imageWidth
        }

        if(parameters.win.height){
            var height = parameters.win.height
        }else{
            var height = textSettings.height + this.padding * 2 + this.getItemPadding()
        }

        return {width, height}
    }

    refreshSideFacePosition(){
        this.faceSprite_textWindow.adjustPosition()
    }

    createAnime(direction = "normal"){
        const {widthAlign, heightAlign} = this.parameters.win.openness
        const {duration, easing, initial, target} = this.parameters.win.position
        const parsedEasing = Eli.Utils.convertEscapeVariablesOnly(easing)
        const [initX, initY] = this.createPosition(initial, target.offsetX, target.offsetY)
        const [targetX, targetY] = this.createPosition(target)
        const defaultData = Eli.AnimeManager.createDefaultData()
        const animationGroupData = {
            onStart: this.onAnimeStart.bind(this),
            onUpdate: this.onAnimeUpdate.bind(this),
            onComplete: this.onAnimeComplete.bind(this, this.getCurrentMessage().text),
        }
        const props = {
            _openness: this.createOpennessAnimeProp(widthAlign, heightAlign),
            x: this.createXAnimeProp(initX, targetX),
            y: this.createYAnimeProp(initY, targetY),
            alpha: this.createAlphaAnimeProp(),
        }

        defaultData.easing = parsedEasing
        defaultData.duration = duration

        this.canPlaySe = true
        this.refreshFaceSprite()

        const animations = Eli.AnimeManager.createAnimations(this, props, defaultData)
        this.animationOpenness = animations[0]
        this.animationX = animations[1]
        this.animationY = animations[2]
        this.animationAlpha = animations[3]
        this.animationGroup = new Eli.AnimeGroup(animations, animationGroupData)
    }

    playAnime(direction = "normal"){
        this.animationGroup.play(direction)
    }

    createPosition(position, targetOffsetX = 0, targetOffsetY = 0){
        const {alignX, alignY, offsetX, offsetY} = position
        const realOffsetX = Number(Eli.Utils.processEscapeVarOrFormula(offsetX))
        const realOffsetY = Number(Eli.Utils.processEscapeVarOrFormula(offsetY))
        const realTargetOffsetX = Number(Eli.Utils.processEscapeVarOrFormula(targetOffsetX))
        const realTargetOffsetY = Number(Eli.Utils.processEscapeVarOrFormula(targetOffsetY))

        const x = {
            left: realOffsetX,
            center: (Graphics.width/2 - this.width/2) + realOffsetX,
            right: (Graphics.width - this.width) + realOffsetX,
            left_offScreen: 0 - (Math.abs(realTargetOffsetX) + this.width),
            right_offScreen: Graphics.width + this.width + Math.abs(realTargetOffsetX),
        }[alignX]

        const y = {
            top: realOffsetY,
            center: (Graphics.height/2 - this.height/2) + realOffsetY,
            bottom: (Graphics.height - this.height) + realOffsetY,
            top_offScreen: 0 - (Math.abs(realTargetOffsetY) + this.height),
            bottom_offScreen: Graphics.height + this.height + Math.abs(realTargetOffsetY),
        }[alignY]
        
        return [Math.round(x), Math.round(y)]
    }

    onAnimeStart(anime){
        if(anime.direction === "normal"){
            this.onAnimeStartNormal(anime)
        }
    }

    onAnimeStartNormal(anime){
        if(!this.isMessageType()){
            this.drawMessageInstantly(this.getCurrentMessage().text)
        }
    }

    onAnimeUpdate(anime){
        if(anime.direction === "reverse"){
            this.onAnimeUpdateReverse()
        }
    }

    onAnimeUpdateReverse(){
        if(this.canResumeOpennessAnime()){
            this.animationOpenness.resume()

            if(this.parameters.win.sound.close.name && this.canPlaySe){
                AudioManager.playSe(this.parameters.win.sound.close)
                this.canPlaySe = false
            }
        }
    }

    canResumeOpennessAnime(){
        return  this.animationX.data.progress >= 80 && 
                this.animationY.data.progress >= 80 && 
                this.animationOpenness.isPaused()
    }

    onAnimeComplete(text, anime){
        if(anime.direction === "normal"){
            this.onAnimeCompleteNormal(text, anime)
        }else{
            this.onAnimeCompleteReverse(text, anime)
        }
    }

    onAnimeCompleteNormal(text, anime){
        if(this.isMessageType() && !this.isStaticReady){
            this.startMessage(text)
        }
    }

    onAnimeCompleteReverse(text, anime){
        if(this.canDestroy()){
            this.destroy()

        }else{
            this.isReadyToUpdate = false

            if(!this.isStaticWindow()){
                this.advanceMessageIndex()
                this.startWindowActivity()
            }
        }
    }

    createOpennessAnimeProp(widthAlign, heightAlign){
        const initialOpenness = (widthAlign === "None" && heightAlign === "None") ? 255 : 0
        return {
            value: [initialOpenness, 255],
            duration: this.parameters.win.openness.duration,
            easing: Eli.Utils.convertEscapeVariablesOnly(this.parameters.win.openness.easing),
            onStart: this.onAnimeOpennessStart.bind(this),
            onUpdate: this.onAnimeOpennessUpdate.bind(this, widthAlign, heightAlign),
            onComplete: this.onAnimeOpennessComplete.bind(this),
        }
    }

    onAnimeOpennessStart(anime){
        if(anime.data.direction.current === "normal"){
            this.onAnimeOpennessStartNormal(anime)
        }else{
            this.onAnimeOpennessStartReverse(anime)
        }
    }

    onAnimeOpennessStartNormal(anime){
        this.animationX.pause()
        this.animationY.pause()
        if(this.parameters.win.sound.open.name){
            AudioManager.playSe(this.parameters.win.sound.open)
        }
    }

    onAnimeOpennessStartReverse(anime){
        anime.pause()
    }
    
    onAnimeOpennessUpdate(widthAlign, heightAlign){
        this.visible = true
        this.updateOpennessDirections(widthAlign, heightAlign)
    }

    updateOpennessDirections(widthAlign, heightAlign){
        this.updateHorizontalOpenness(widthAlign)
        this.updateVerticalOpenness(heightAlign)
    }
    
    updateHorizontalOpenness(widthAlign){
        switch(widthAlign){
            case "Left to Right":
                this._container.scale.x = this._openness / 255
                this._container.x = (1 - this._openness / 255)
                break
            case "Centered":
                this._container.scale.x = this._openness / 255
                this._container.x = (this.width / 2) * (1 - this._openness / 255)
                break
            case "Right to Left":
                this._container.scale.x = this._openness / 255
                this._container.x = this.width * (1 - this._openness / 255)
                break
            default:
                this._container.scale.x = 1
                this._container.x = 0
        }
    }
    
    updateVerticalOpenness(heightAlign){
        switch(heightAlign){
            case "Top to Bottom":
                this._container.scale.y = this._openness / 255
                this._container.y = (1 - this._openness / 255)
                break
            case "Centered":
                this._container.scale.y = this._openness / 255
                this._container.y = (this.height / 2) * (1 - this._openness / 255)
                break
            case "Bottom to Top":p
                this._container.scale.y = this._openness / 255
                this._container.y = this.height * (1 - this._openness / 255)
                break
            default:
                this._container.scale.y = 1
                this._container.y = 0
        }
    }
    
    onAnimeOpennessComplete(anime){
        if(anime.data.direction.current === "normal"){
            this.onAnimeOpennessCompleteNormal(anime)
        }
    }

    onAnimeOpennessCompleteNormal(anime){
        this.animationX.play()
        this.animationY.play()
    }

    createXAnimeProp(initX, targetX){
        return {
            value: [initX, targetX],
            onUpdate: this.onAnimeXUpdate.bind(this),
        }
    }
    
    onAnimeXUpdate(anime){
        this.x = Math.round(this.x)
    }
    
    createYAnimeProp(initY, targetY){
        return {
            value: [initY, targetY],
            onUpdate: this.onAnimeYUpdate.bind(this)
        }
    }
    
    onAnimeYUpdate(anime){
        this.y = Math.round(this.y)
    }

    createAlphaAnimeProp(){
        const [initial, target, duration] = this.parameters.win.opacity
        return {
            value: [initial, target],
            duration: duration,
        }
    }

    refreshFaceSprite(){
        this.faceSprite_textWindow.refreshFaceBitmap(this.getCurrentMessage().faceName, this.getCurrentMessage().faceIndex) 
    }

    isMessageType(){
        return this.parameters.contents.writeType === "Message"
    }

    drawMessageInstantly(text){
        this.startMessage(text)
        this.processAllText(this._textState)
        this.onEndOfText()
    }

/* ------------------------------ TEXT SECTION ------------------------------ */

    startMessage(text){
        this._textState = this.createTextState(text, 0, 0, 0)
        this.newPage(this._textState)
    }

    createTextState(text, x, y, width){
        const textState = super.createTextState(text, x, y, width)
        textState.x = this.newLineX(textState)
        textState.startX = textState.x

        return textState
    }

    newLineX(textState){
        let margin = 0

        if(this.hasFaceImage()){
            margin = this.newLineXForFace()

        }else if(this.hasCharImage()){
            margin = this.newLineXForChar()
        }

        const x = textState.rtl ? this.innerWidth - margin : margin

        return x + this.parameters.contents.textOffsetX

    }

    hasFaceImage(){
        return this.imageTypeIsFace() && this.faceSprite_textWindow.faceName !== ""
    }

    imageTypeIsFace(){
        return this.parameters.contents.image.type === "Face"
    }

    newLineXForFace(){
        const width = Eli.Utils.getFaceSize().width
        const spacing = 20

        return width + spacing
    }

    hasCharImage(){
        return this.imageTypeIsCharacter() && this.character_textWindow && ( this.character_textWindow.characterName() || this.character_textWindow.tileId() )
    }

    imageTypeIsCharacter(){
        return this.parameters.contents.image.type === "Character"
    }

    newLineXForChar(){
        const width = this.character_textWindow.getMapSprite().patternWidth()
        const spacing = 10

        return width + spacing
    }

    onEndOfText(){
        this._textState = null
        this.isReadyToUpdate = false

        if(this.isStaticWindow()){
            this.isStaticReady = true
        }
        
    }

    needsNewPage(textState){
        /* 
            It never needs a new page. The window only have one page.
            When it writes again, it clears all content. 
        */
        return false
    }

/* ---------------------------- END TEXT SECTION ---------------------------- */

    canDestroy(){
        return !this.hasMessageToWrite() && !this.isStaticWindow()
    }

    hasMessageToWrite(){
        return (this.messageIndex + 1) < this.messages.length
    }

/* --------------------- INHERIT FROM ELI ANIMATED FACES -------------------- */

    updateIdleFaceAnimation(){
        this.faceSprite_textWindow.updateAnimation(this.faceSprite_textWindow.settings.middleIndex)
    }

    updateTalkingFaceAnimation(){
        this.faceSprite_textWindow.updateAnimation(this.faceSprite_textWindow.settings.endIndex)
    }

    createFaceSprite(){}

/* ------------------ OVERWRITE FROM DEFAULT MESSAGE WINDOW ----------------- */

    updatePlacement(){}
    createSubWindows(){}
    updateSpeakerName(){}
    loadMessageFace(){}
    terminateMessage(){}

}

Eli.TextWindow = {

    version: 6.01,
    url: "https://hakuenstudio.itch.io/hakuen-studio-text-window-for-rpg-maker-mv-mz",
    parameters: {
        templates: [{
            id: "",
            isStatic: false,
            updateInterval: 0,
            messages: [{
                faceName: "",
                faceIndex: 0,
                text: "",
                spriteId: 0,
                charName: "",
                charIndex: 0,
            }],
            contents: {
                image: {
                    type: "",
                    alignY: "",
                    layer: "",
                },
                textOffsetX: 0,
                writeType: "",
            },
            win: {
                backgroundType: "",
                height: 0,
                layer: "",
                skin: "",
                tone: [0, 0, 0],
                width: 0,
                opacity: [0, 0, 0],
                position: {
                    initial: {
                        alignX: "",
                        offsetX: 0,
                        alignY: "",
                        offsetY: 0,
                    },
                    target: {
                        alignX: "",
                        offsetX: 0,
                        alignY: "",
                        offsetY: 0,
                    },
                    easing: "",
                    duration: 0,
                    delay: 0,
                },  
                openness: {
                    widthAlign: "none", 
                    heightAlign: "center",
                    duration: 0,
                    easing: "inherit",
                },
                sound: {
                    open: {name: "", pan: 0, pitch: 0, volume: 0},
                    close: {name: "", pan: 0, pitch: 0, volume: 0},
                },
            },
        }],
    },
    alias: {},
    tempChar: {
        id: 0,
        name: "",
        index: -1,
    },
    currentTemplateId: null,
    eventIntegration: false,
    needPreloadImagesForText: false,
    staticWindows: {},
    Container_TextWindow: Container_TextWindow,
    Sprite_TextWindowFace: Sprite_TextWindowFace,
    Sprite_TextWindowAnimatedFace: Sprite_TextWindowAnimatedFace,
    Sprite_TextWindowCharacter: Sprite_TextWindowCharacter,
    Window_TextWindow: Window_TextWindow,

    initialize(){
        this.initParameters()
        this.initPluginCommands()
    },

    initParameters(){
        this.parameters = PluginManager.parameters("EliMZ_TextWindow")
        this.parameters.templates = JSON.parse(this.parameters.templates)

        for(let i = 0; i < this.parameters.templates.length; i++){
            const template = this.parseTemplateParameters(this.parameters.templates[i])

            this.parameters.templates[i] = template
        }
    },

    parseTemplateParameters(parameters){
        const template = JSON.parse(parameters)

        template.id = template.id.toLowerCase()
        template.messages = []
        template.isStatic = template.isStatic === "true"
        template.updateInterval = template.updateInterval === "Manual" ? Infinity : Number(template.updateInterval)
        template.win = this.parseWinParameters(template.win)
        template.contents = this.parseContentParameters(template.contents)

        return template
    },

    parseWinParameters(parameters){
        const win = JSON.parse(parameters)
        win.skin = win.skin
        win.backgroundType = this.findBackgroundType(win.backgroundType)
        win.tone = this.formatTone(win.tone)
        win.layer = win.layer // Can be Variable!
        win.width = Number(win.width)
        win.height = Number(win.height)
        win.position = this.parsePositionParameters(win.position)
        win.openness = this.parseOpennessParameters(win.openness, win.position.easing)
        win.sound = this.parseSoundParameters(win.sound)
        win.opacity = this.formatOpacity(win.opacity)

        return win
    },

    findBackgroundType(type){
        const options = {
            "Window":                   0,
            "Dim":                      1,
            "Transparent":              2,
            "Strong":                   3,
            "Light Gradient Vertical":  4,
            "Faded Horizontal":         5,
        }
        return options[type]
    },

    formatTone(tone){
        return tone.split(",").map(item => Number(item))
    },

    parsePositionParameters(parameters){
        const position = JSON.parse(parameters)
        position.duration = Number(position.duration)
        position.easing = position.easing
        position.delay = Number(position.delay)

        position.initial = {
            alignX: position.initialAlignX,
            offsetX: position.initialOffsetX,
            alignY: position.initialAlignY,
            offsetY: position.initialOffsetY,       
        }
        position.target = {
            alignX: position.targetAlignX,
            offsetX: position.targetOffsetX,
            alignY: position.targetAlignY,
            offsetY: position.targetOffsetY,       
        }

        delete position.initialAlignX 
        delete position.initialOffsetX
        delete position.initialAlignY
        delete position.initialOffsetY
        delete position.targetAlignX
        delete position.targetOffsetX
        delete position.targetAlignY
        delete position.targetOffsetY

        return position
    },

    parseOpennessParameters(parameters, easingPosition){
        const openness = JSON.parse(parameters)

        if(openness.easing === "inherit"){
            openness.easing = easingPosition
        }else{
            openness.easing = openness.easing
        }
        
        openness.duration = Number(openness.duration)

        return openness
    },

    parseSoundParameters(parameters){
        const sound = JSON.parse(parameters)
        sound.close = JSON.parse(sound.close)
        sound.close.name = sound.close.name
        sound.close.pan = Number(sound.close.pan)
        sound.close.volume = Number(sound.close.volume)
        sound.close.pitch = Number(sound.close.pitch)

        sound.open = JSON.parse(sound.open)
        sound.open.name = sound.open.name
        sound.open.pan = Number(sound.open.pan)
        sound.open.volume = Number(sound.open.volume)
        sound.open.pitch = Number(sound.open.pitch)

        return sound
    },

    formatOpacity(opacity){
        const arr = opacity.split(",").map(item => Number(item))
        return [
            arr[0]/255,
            arr[1]/255,
            arr[2],
        ]
    },

    parseContentParameters(parameters){
        const contents = JSON.parse(parameters)
        contents.textOffsetX = Number(contents.textOffsetX)
        contents.writeType = contents.writeType
        contents.image = {
            type: contents.imageType,
            alignY: contents.imageAlignY,
            layer: contents.imageLayer,
        }

        delete contents.imageType
        delete contents.imageAlignY
        delete contents.imageLayer
        
        return contents
    },

    initPluginCommands(){
        const commands = ["cmd_setupAndShow", "cmd_staticWindow", "cmd_refreshStatic"]
        Eli.PluginManager.registerCommands(this, commands)
    },

    cmd_setupAndShow(args){
        const templateId = Eli.Utils.convertEscapeVariablesOnly(args.id.toLowerCase())
        const winId = Eli.Utils.convertEscapeVariablesOnly( (args.winId || "").toLowerCase() )

        let template = this.parameters.templates.find(template => template.id.toLowerCase() === templateId)

        if(!template){
            template = this.parseTemplateParameters(args.template)
            this.parameters.templates.push(template)
        }
        
        if(template){

            template.messages = this.parseMessageParameters(args.messages)
            this.currentTemplateId = null
            this.eventIntegration = false

            if(template.messages.length > 0){

                if(this.needPreloadImagesForText){
                    setTimeout(() => {this.createTextWindow(template, winId)}, 100)

                }else{
                    this.createTextWindow(template, winId)
                }
                
            }
            this.needPreloadImagesForText = false
            
        }else{
            console.log(`Eli Text Window: Template not found`)
        }
    },

    cmd_staticWindow(args){
        const id = args.winId.toLowerCase()
        const win = this.staticWindows[id]

        if(win){
            const messages = this.parseMessageParameters(args.messages)
            const opacity = args.opacity ? Number(args.opacity) : win.opacity
            const x = args.x ? Number(Eli.Utils.convertEscapeVariablesOnly(args.x)) : win.x
            const y = args.y ? Number(Eli.Utils.convertEscapeVariablesOnly(args.y)) : win.y
            const action = args.action
            const rawIndex = args.index

            if(action === "Close"){
                win.playAnime("reverse")

            }else if(action === "Open"){
                win.playAnime("normal")
            }

            win.opacity = opacity

            if(args.positionOperation === "Set"){
                win.x = x
                win.y = y

            }else{
                win.x += x
                win.y += y
            }

            win.messages.push(...messages)
            this.updateStaticWindowMessage(win, rawIndex)
        
        }else{
            console.log(`There is no window called ${id}`)
        }
        
    },

    cmd_refreshStatic(args){
        const winId = args.winId.toLowerCase()

        if(this.staticWindows[winId]){
            this.staticWindows[winId].refreshStaticText()
        }
    },

    updateStaticWindowMessage(win, rawIndex){
        if(rawIndex === "None"){
            return;

        }else if(rawIndex === "Next"){
            const index = Number(win.messageIndex) + 1
            win.changeMessage(index)

        }else if(rawIndex === "Previous"){
            const index = Number(win.messageIndex) - 1
            win.changeMessage(index)

        }else if(rawIndex === "Last"){
            const index = win.messages.lenght-1
            win.changeMessage(index)

        }else{
            const index = Number(rawIndex)
            win.changeMessage(index)
        }
    },

    createDummyWin(){
        if(Utils.RPGMAKER_NAME === "MV"){
            return new Window_Base(0, 0, 200, 200)
        }else{
            return new Window_Base(new Rectangle(0, 0, 200, 200))
        }
    },

    parseMessageParameters(rawMessages){
        const messages = JSON.parse(rawMessages)
        const dummyWin = this.createDummyWin()

        for(let i = 0; i < messages.length; i++){
            const msg = JSON.parse(messages[i])
            msg.charIndex = msg.charIndex
            msg.faceIndex = msg.faceIndex
            msg.spriteId = msg.spriteId == 0 ? Eli.PluginManager.currentEventId : Number(Eli.Utils.convertEscapeVariablesOnly(msg.spriteId))

            if(msg.text.toLowerCase().includes("drawimg")){
                this.needPreloadImagesForText = true
                dummyWin.drawTextEx(msg.text.substring(0)) // Preload images that has "drawimg" escape code.
            }
            
            delete msg.separator1
            delete msg.separator2
            messages[i] = msg
        }
        
        return messages
    },

    createTextWindow(template, winId){
        const rect = new Rectangle(0, 0, 1000, 1000)
        const textWindow = new Window_TextWindow(rect, template, winId)
        const layer = Eli.Utils.convertEscapeVariablesOnly(template.win.layer)

        this.addTextWindowByLayer(layer, textWindow)
        textWindow.startWindowActivity()  
    },

    addTextWindowByLayer(layer, win){
        const scene = SceneManager._scene
        const Container = {
            "Below Pictures": scene._spriteset.textWindowContainer_belowPictures,
            "Above Pictures and below Windows": scene._spriteset.textWindowContainer_abovePictures,
            "Above Windows": scene.textWindowContainer_aboveWindows,
        }[layer]

        const oldWinIndex = Container.children.findIndex(child => child.winId === win.winId)

        if(oldWinIndex > -1){
            const oldWin = Container.children[oldWinIndex]
            Container.removeChildAt(oldWinIndex)
            oldWin.destroy()
        }

        Container.addChild(win)
    },

    param(){
        return this.parameters
    },

    resetTempChar(){
        this.tempChar = {
            id: 0,
            name: "",
            index: -1,
        }
    },
}

const Plugin = Eli.TextWindow
const Alias = Eli.TextWindow.alias

Plugin.initialize()

/* ------------------------------- SCENE BASE ------------------------------- */
{

Alias.Scene_Base_initialize = Scene_Base.prototype.initialize
Scene_Base.prototype.initialize = function(){
    Alias.Scene_Base_initialize.call(this)
    Plugin.staticWindows = {}
}

Alias.Scene_Base_create = Scene_Base.prototype.create
Scene_Base.prototype.create = function() {
    Alias.Scene_Base_create.call(this)
    this.createTextWindowContainer()
}

Scene_Base.prototype.createTextWindowContainer = function(){
    this.textWindowContainer_aboveWindows = new Container_TextWindow()
    this.addChild(this.textWindowContainer_aboveWindows) 
}

Scene_Base.prototype.setTextWindowAboveWindowLayer = function(){
    this.setChildIndex(this.textWindowContainer_aboveWindows, this.children.length-1) 
}

Alias.Scene_Base_start = Scene_Base.prototype.start
Scene_Base.prototype.start = function(){
    this.setTextWindowAboveWindowLayer()
    Alias.Scene_Base_start.call(this)
}

}

/* ----------------------------- SPRITESET BASE ----------------------------- */
{

Alias.Spriteset_Base_createUpperLayer = Spriteset_Base.prototype.createUpperLayer
Spriteset_Base.prototype.createUpperLayer = function() {
    this.createTextWindowBelowPictures()
    Alias.Spriteset_Base_createUpperLayer.call(this)
    this.createTextWindowAbovePictures()
}

Spriteset_Base.prototype.createTextWindowBelowPictures = function() {
    this.textWindowContainer_belowPictures = new Container_TextWindow()
    this.addChild(this.textWindowContainer_belowPictures)
}

Spriteset_Base.prototype.createTextWindowAbovePictures = function() {
    this.textWindowContainer_abovePictures = new Container_TextWindow()
    this.addChild(this.textWindowContainer_abovePictures)
}

}

/* ---------------------------- GAME INTERPRETER ---------------------------- */
{

// Show Text
Alias.Game_Interpreter_command101 = Game_Interpreter.prototype.command101
Game_Interpreter.prototype.command101 = function(params){
    if(Plugin.eventIntegration){
        const template = Plugin.parameters.templates.find(item => item.id === Plugin.currentTemplateId)

        if(template){
            this.command101_setupTextWindowMessage(params, template)
        }

        return true
    }else{
        return Alias.Game_Interpreter_command101.call(this, params)
    }
}

Game_Interpreter.prototype.command101_setupTextWindowMessage = function(params, template) {
    const msgIndex = template.messages.length
    const {charName, spriteId, charIndex} = this.createTextWindowCharSettings()
    template.messages.push({
        charIndex: charIndex,
        charName: charName,
        faceIndex: params[1],
        faceName: params[0],
        spriteId: spriteId,
        text: "",
    })
    Plugin.resetTempChar()

    while(this.nextEventCode() === 401){
        this._index++
        const cmdText = this.currentCommand().parameters[0]
        template.messages[msgIndex].text += template.messages[msgIndex].text ? "\n" + cmdText : cmdText
    }
}

Game_Interpreter.prototype.createTextWindowCharSettings = function() {
    let {name, id, index} = Plugin.tempChar
    name = Eli.Utils.convertEscapeVariablesOnly(name)
    id = Eli.Utils.convertEscapeVariablesOnly(String(id))
    index = Number(Eli.Utils.convertEscapeVariablesOnly(String(index)))

    const spriteId = Number(id) || id || this._eventId
    const sprite = Eli.Utils.getSpriteCharacter(spriteId)
    const character = sprite._character
    const charName = name || character.characterName()
    const charIndex = index > -1 ? index : character.characterIndex()

    return {spriteId, charName, charIndex}
}

}

}